/***************************************************************************
 *
 * Copyright 2010,2011 BMW Car IT GmbH
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "configurator.h"
#include "ICommandExecutor.h"
#include "CommitCommand.h"
#include "Configuration.h"
#include "Log.h"
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <dirent.h>

#include "pdd.h"
#include "pdd_config_nor_user.h"
#include "dispvidctrl_datapool.h"
#include "VD_EARLYNorData.h"

AIVIConfigurator::AIVIConfigurator(ICommandExecutor& executor, Configuration& config)
: IConfigurator(&executor)
, mExecutor(executor)
, mConfiguration(config)
{
    tU8EarlyBrand = 0;
    tU8EarlyDisplayOrientation = 0;
    memset(tU8EarlyResolution, 0, sizeof(tU8EarlyResolution));
    memset(tU8TrDisplayOrder, 0, sizeof(tU8TrDisplayOrder));
    tU8TrDisplayOrder[0] = 0;
    tU8TrDisplayOrder[1] = 2;
    tU8TrDisplayOrder[2] = 1;
    tU8TrDisplayOrder[3] = 3;
    memset(tU8TrFlushDesignConfigFile, 0, sizeof(tU8TrFlushDesignConfigFile));
	
#ifdef VARIANT_S_FTR_ENABLE_WHITEPOINT_CORRECTION
	// AIVI-68581: Whitepoint correction
	iWPC_ScreenID = (unsigned int)ILM_DISPLAY_UNKNOWN;
	for (int i = 0; (sizeof(iaWPC_Diagnal_matrix)/sizeof(iaWPC_Diagnal_matrix[0])) > i; ++i)
	{
		iaWPC_Diagnal_matrix[i] = ILM_INVALID_CSC_MATRIX_VALUE;
	}
#endif //VARIANT_S_FTR_ENABLE_WHITEPOINT_CORRECTION
}

/* has to be the same for all Configurator plugins*/
PluginApi AIVIConfigurator::pluginGetApi() const
{
    return Configurator_Api_v1;
}

/* can be the same for all Scene plugins
 * this plugin is not running all the time
 * nobody should use this function
 * but we have to fulfill the plugin interface*/
HealthCondition AIVIConfigurator::pluginGetHealth()
{
    return HealthRunning;
}

/*Core function of the Scene plugin
 * this will be executed when plugin is loaded
 * all Scene setup has to be defined here*/
bool AIVIConfigurator::configure()
{
    /*read out the configuration from NOR early*/
    bool success = vReadEarlyConfig();
    char str[200] = {0};

    if(success == true) {
        /*set new path for input device configuration file*/
        mConfiguration.setInputDevConfigPath((const char*)tU8TrFlushDesignConfigFile);

        /*set new mapping for display id's, always provide a mapping for 4 possible display
        * even they are not used, duplicated are not allowed*/
        mConfiguration.setDisplayIDmapping(tU8TrDisplayOrder[0], tU8TrDisplayOrder[1], tU8TrDisplayOrder[2], tU8TrDisplayOrder[3]);

#ifdef VARIANT_S_FTR_ENABLE_WHITEPOINT_CORRECTION
        // AIVI-68581: Whitepoint correction
        sprintf(str, "AIVIConfigurator::configure(): Whitepoint correction for screen_id %d is [%d, %d, %d]",
        		iWPC_ScreenID, iaWPC_Diagnal_matrix[0], iaWPC_Diagnal_matrix[1], iaWPC_Diagnal_matrix[2]);
        LOG_INFO("AIVIConfigurator", str);
        if (ILM_DISPLAY_MAX_NUM > (e_ilmDisplays)iWPC_ScreenID)
        {
            ilmCSCProperties _tCSCProperties = { true, // set_to_default
            		0, // hue
					0, // saturation
					0, // brightness
					0, // contrast
					0, // hue_off
					0, // saturation_off
					0, // brightness_off
					ILM_CSC_TYPE_RGB2RGB  // type
            };
            ilmUSERSpecificCSCvalues tUserCSC_Values = {
            		{iaWPC_Diagnal_matrix[0], iaWPC_Diagnal_matrix[1], iaWPC_Diagnal_matrix[2]},
					{ILM_INVALID_CSC_MATRIX_VALUE, ILM_INVALID_CSC_MATRIX_VALUE, ILM_INVALID_CSC_MATRIX_VALUE}
            };
            ilmUSERSpecificCSCvalues tUserCSC_Values_UnusedScreen = {
            		{ILM_INVALID_CSC_MATRIX_VALUE, ILM_INVALID_CSC_MATRIX_VALUE, ILM_INVALID_CSC_MATRIX_VALUE},
					{ILM_INVALID_CSC_MATRIX_VALUE, ILM_INVALID_CSC_MATRIX_VALUE, ILM_INVALID_CSC_MATRIX_VALUE}
            };
            bool success_setcscvalue = false;
            unsigned int _ScreenID = iWPC_ScreenID;
            ilmUSERSpecificCSCvalues* _potUserCSC_Values_Screen1 = (1 == iWPC_ScreenID) ? &tUserCSC_Values: &tUserCSC_Values_UnusedScreen;
            ilmUSERSpecificCSCvalues* _potUserCSC_Values_Screen2 = (2 == iWPC_ScreenID) ? &tUserCSC_Values: &tUserCSC_Values_UnusedScreen;

            // set Screen 1
            _ScreenID = 1;
            success_setcscvalue = mConfiguration.setCSCValues(_ScreenID, &_tCSCProperties);
            sprintf(str, "AIVIConfigurator::configure(): setCSCValues for screen_id %d is [set2default: %d, hue: %d, brightness: %d, contrast: %d, hue_off: %d, saturation_off: %d, brightness_off: %d] returns %d",
            		_ScreenID, _tCSCProperties.set_to_default, _tCSCProperties.hue, _tCSCProperties.saturation, _tCSCProperties.brightness, _tCSCProperties.contrast,
					_tCSCProperties.hue_off, _tCSCProperties.saturation_off, _tCSCProperties.brightness_off, success_setcscvalue);
            LOG_INFO("AIVIConfigurator", str);
            success_setcscvalue = mConfiguration.setCSCValues(_ScreenID, NULL, _potUserCSC_Values_Screen1);
            sprintf(str, "AIVIConfigurator::configure(): setCSCValues for screen_id %d is [%d, %d, %d] returns %d",
            		_ScreenID, _potUserCSC_Values_Screen1->diagnal_matrix[0], _potUserCSC_Values_Screen1->diagnal_matrix[1], _potUserCSC_Values_Screen1->diagnal_matrix[2], success_setcscvalue);
            LOG_INFO("AIVIConfigurator", str);
            // set Screen 2
            _ScreenID = 2;
            success_setcscvalue = mConfiguration.setCSCValues(_ScreenID, &_tCSCProperties);
            sprintf(str, "AIVIConfigurator::configure(): setCSCValues for screen_id %d is [set2default: %d, hue: %d, brightness: %d, contrast: %d, hue_off: %d, saturation_off: %d, brightness_off: %d] returns %d",
            		_ScreenID, _tCSCProperties.set_to_default, _tCSCProperties.hue, _tCSCProperties.saturation, _tCSCProperties.brightness, _tCSCProperties.contrast,
					_tCSCProperties.hue_off, _tCSCProperties.saturation_off, _tCSCProperties.brightness_off, success_setcscvalue);
            LOG_INFO("AIVIConfigurator", str);
            success_setcscvalue = mConfiguration.setCSCValues(_ScreenID, NULL, _potUserCSC_Values_Screen2);
            sprintf(str, "AIVIConfigurator::configure(): setCSCValues for screen_id %d is [%d, %d, %d] returns %d",
            		_ScreenID, _potUserCSC_Values_Screen2->diagnal_matrix[0], _potUserCSC_Values_Screen2->diagnal_matrix[1], _potUserCSC_Values_Screen2->diagnal_matrix[2], success_setcscvalue);
            LOG_INFO("AIVIConfigurator", str);
        }
#endif //VARIANT_S_FTR_ENABLE_WHITEPOINT_CORRECTION

        updateScreenLayoutsDefinition();
    }

    LOG_INFO("AIVIConfigurator", "CONFIGURATOR PLUGIN COMPLETE");

    return true;
}

/* should be the name of a class!!!*/
t_ilm_const_string AIVIConfigurator::pluginGetName() const
{
    return "AIVIConfigurator";
}

int AIVIConfigurator::getIterationCounter()
{
    // this plugin is one-shot and is not monitored.
    static int dummyValue = 0;
    return ++dummyValue;
}

bool AIVIConfigurator::vReadEarlyConfig() {

    // Set default values
    tU8EarlyBrand = 0;
    tU8EarlyDisplayOrientation = 0;
    tU8EarlyResolution[0] = '\0';
    tU8TrDisplayOrder[0] = 0;
    tU8TrDisplayOrder[1] = 2;
    tU8TrDisplayOrder[2] = 1;
    tU8TrDisplayOrder[3] = 3;
    tU8TrFlushDesignConfigFile[0] = '\0';

    char str[200];

    //Get settings data from NOR data
    int s32StreamLength = 800;

    bool bSuccess = true;

    TEarlyNorForceConfiguration tEarlyNorForce;

    //Allocate buffer
    unsigned char* pu8DataStreamBuffer = (unsigned char*) malloc((unsigned int)s32StreamLength);
    tU8            Vu8Info             = PDD_READ_INFO_NORMAL_FILE;
    int s32Result = pdd_read_datastream_early_from_nor((const char*)PDD_NOR_USER_DATASTREAM_NAME_DATASET_VD_EARLY_FORCE_CONFIG, pu8DataStreamBuffer, s32StreamLength, 1, &Vu8Info);
    if (s32Result > 0) {
        if(pdd_helper_get_element_from_stream((const char *)"EarlyNorConfigForce",pu8DataStreamBuffer,(size_t)s32Result, (void*) &tEarlyNorForce, sizeof(TEarlyNorForceConfiguration),0)==0) { //lint !e605
            sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() Brand READ FROM NOR %d", tEarlyNorForce.u8EarlyBrand);
            LOG_INFO("AIVIConfigurator", str);
            sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() DisplayOrientation READ FROM NOR %d", tEarlyNorForce.u8EarlyDisplayOrientation);
            LOG_INFO("AIVIConfigurator", str);
            sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() Resolution READ FROM NOR %s", tEarlyNorForce.au8EarlyResolution);
            LOG_INFO("AIVIConfigurator", str);
            sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() TrDisplayOrder READ FROM NOR %d, %d, %d, %d", tEarlyNorForce.au8EarlyDisplayOrder[0], tEarlyNorForce.au8EarlyDisplayOrder[1], tEarlyNorForce.au8EarlyDisplayOrder[2], tEarlyNorForce.au8EarlyDisplayOrder[3]);
            LOG_INFO("AIVIConfigurator", str);
            sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() TrFlushDesignConfigFile READ FROM NOR %s", tEarlyNorForce.au8EarlyFlushDesignConfigFile);
            LOG_INFO("AIVIConfigurator", str);
            tU8EarlyBrand = tEarlyNorForce.u8EarlyBrand;
            tU8EarlyDisplayOrientation = tEarlyNorForce.u8EarlyDisplayOrientation;
            memcpy(&tU8EarlyResolution[0], &tEarlyNorForce.au8EarlyResolution[0], sizeof tU8EarlyResolution);
			tU8TrDisplayOrder[0] = tEarlyNorForce.au8EarlyDisplayOrder[0];
			tU8TrDisplayOrder[1] = tEarlyNorForce.au8EarlyDisplayOrder[1];
			tU8TrDisplayOrder[2] = tEarlyNorForce.au8EarlyDisplayOrder[2];
			tU8TrDisplayOrder[3] = tEarlyNorForce.au8EarlyDisplayOrder[3];
            memcpy(&tU8TrFlushDesignConfigFile[0], &tEarlyNorForce.au8EarlyFlushDesignConfigFile[0], sizeof tU8TrFlushDesignConfigFile);
        }
        else {
            bSuccess = false;
            LOG_ERROR("AIVIConfigurator", "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() EarlyNorConfigForce READ FROM NOR Failure");
        }
    }
    else {
        bSuccess = false;
        sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() pdd_read_datastream_early_from_nor Failure %d", s32Result);
        LOG_WARNING("AIVIConfigurator", str);
    }

#ifdef VARIANT_S_FTR_ENABLE_WHITEPOINT_CORRECTION
    // AIVI-68581: Whitepoint correction
	iWPC_ScreenID = (unsigned int)ILM_DISPLAY_UNKNOWN;
	for (int i = 0; (sizeof(iaWPC_Diagnal_matrix)/sizeof(iaWPC_Diagnal_matrix[0])) > i; ++i)
	{
		iaWPC_Diagnal_matrix[i] = ILM_INVALID_CSC_MATRIX_VALUE;
	}
    if (NULL != pu8DataStreamBuffer)
    {
        memset(pu8DataStreamBuffer, 0, s32StreamLength);
    	s32Result = pdd_read_datastream_early_from_nor((const char*)PDD_NOR_USER_DATASTREAM_NAME_EARLYCONFIGLAYERMANAGER, pu8DataStreamBuffer, s32StreamLength, 1, &Vu8Info);
    	if (0 < s32Result)
    	{
        	tU8 _u8DpElementVersion = 0x00;
        	const tS32 _s32ReadBufferSize = s32Result;
        	// reading ScreenID ... datapool element TrWpcScreenID
        	tU8 _u8WpcScreenID = 0;
        	tString _strWpcScreenID = "TrWpcScreenID";
        	s32Result = pdd_helper_get_element_from_stream(_strWpcScreenID, (const void*)pu8DataStreamBuffer, _s32ReadBufferSize,
        			(void*)(&_u8WpcScreenID), sizeof(_u8WpcScreenID), _u8DpElementVersion);
        	if (0 == s32Result)
        	{
                sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfig() EXTRACTED %s is %d", _strWpcScreenID, _u8WpcScreenID);
                LOG_INFO("AIVIConfigurator", str);
        		iWPC_ScreenID = (ILM_DISPLAY_MAX_NUM > _u8WpcScreenID)?(unsigned int)_u8WpcScreenID:(unsigned int)ILM_DISPLAY_UNKNOWN;
        	}
        	else
        	{
        		// datapool element not found
        		iWPC_ScreenID = (unsigned int)ILM_DISPLAY_UNKNOWN;
                sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfig() returns %d during extraction of %s (use %d)", s32Result, _strWpcScreenID, iWPC_ScreenID);
                LOG_ERROR("AIVIConfigurator", str);
        	}
        	// reading RGB ... datapool element TrWpcRGB
        	tU8 _au8WpcRGB[sizeof(iaWPC_Diagnal_matrix)/sizeof(iaWPC_Diagnal_matrix[0])] = {0};
            memset(_au8WpcRGB, 0, sizeof(_au8WpcRGB));
        	tString _strWpcRGB = "TrWpcRGB";
        	s32Result = pdd_helper_get_element_from_stream(_strWpcRGB, (const void*)pu8DataStreamBuffer, _s32ReadBufferSize,
        			(void*)&_au8WpcRGB, sizeof(_au8WpcRGB), _u8DpElementVersion);
        	if (0 == s32Result)
        	{
        		for (int i=0; (sizeof(iaWPC_Diagnal_matrix)/sizeof(iaWPC_Diagnal_matrix[0])) > i; ++i)
        		{
                    sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfig() EXTRACTED %s[%d] is %d", _strWpcRGB, i, _au8WpcRGB[i]);
                    LOG_INFO("AIVIConfigurator", str);
        			iaWPC_Diagnal_matrix[i] = (t_ilm_int)_au8WpcRGB[i];
        		}
        	}
        	else
        	{
        		// datapool element not found
        		for (int i = 0; (sizeof(iaWPC_Diagnal_matrix)/sizeof(iaWPC_Diagnal_matrix[0])) > i; ++i)
        		{
        			iaWPC_Diagnal_matrix[i] = (t_ilm_int)ILM_INVALID_CSC_MATRIX_VALUE;
        		}
                sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfig() returns %d during extraction of %s (use default)", s32Result, _strWpcRGB);
                LOG_ERROR("AIVIConfigurator", str);
        	}
    	}
    	else
    	{
    		// EarlyConfig_LayerManager data unavailable
    		sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfig() PDD_NOR_USER_DATASTREAM_NAME_EARLYCONFIGLAYERMANAGER Failure %d", s32Result);
            LOG_WARNING("AIVIConfigurator", str);
    	}
    }
#endif //VARIANT_S_FTR_ENABLE_WHITEPOINT_CORRECTION

    free(pu8DataStreamBuffer);
	
	// Always return success - as returning fail will stop the target starting during production 
	// Note - than when there is a failure, an error message is added to the log
    return true;
}

void AIVIConfigurator::updateScreenLayoutsDefinition()
{
   if(createScreenLayoutsDir())
   {
       ::std::string name = "/opt/bosch/hmibase/" + getScreenLayoutsName();
	   
       char str[200];
       sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::updateScreenLayoutsDefinition() Screen Layout file name %s", name.c_str());
       LOG_INFO("AIVIConfigurator", str);
	   
       if (isFileExistsInSystem(name))
       {
           mode_t lastMode   = umask(0);
           ::std::string cmd = "ln -sf ";
           cmd.append(name);
           cmd.append(" /tmp/hmi/ScreenLayouts/ScreenLayouts.xml");
           if (system(cmd.c_str()) != 0)
           {
              LOG_INFO("AIVIConfigurator", "CONFIGURATOR PLUGIN - AIVIConfigurator::updateScreenLayoutsDefinition() Screen layout symlink update failed: Symlink CMD is not executed");
           }
           umask(lastMode);
       }
       else
       {
           LOG_INFO("AIVIConfigurator", "CONFIGURATOR PLUGIN - AIVIConfigurator::updateScreenLayoutsDefinition() Screen layout symlink update failed: Invalid File");
       }
   }
   else
   {
      LOG_INFO("AIVIConfigurator", "CONFIGURATOR PLUGIN - AIVIConfigurator::updateScreenLayoutsDefinition() Screen layout symlink update failed: Invalid ScreenLayout Dir");
   }
}

bool AIVIConfigurator::createScreenLayoutsDir() const
{
    bool isValid = false;
    DIR* pDir = opendir("/tmp/hmi/ScreenLayouts");
    if (NULL == pDir)
    {
        if (system("mkdir -p /tmp/hmi/ScreenLayouts") == 0)
        {
           isValid = true;
        }
    }
    else
    {
       isValid = true;
       closedir(pDir);
    }
    return isValid;
}

bool AIVIConfigurator::isFileExistsInSystem(const ::std::string& name) const
{
    bool isValid = false;
    if (!name.empty())
    {
        struct stat buffer;
        if ((stat(name.c_str(), &buffer) == 0) && (S_ISREG(buffer.st_mode)))
        {
            isValid = true;
        }
    }
    return isValid;
}

::std::string AIVIConfigurator::getScreenLayoutsName() const
{
    ::std::string name = "ScreenLayouts_rnaivi.xml";
    switch (tU8EarlyBrand)
    {
        case DISPVIDCTRL_AllianceBrandType_Renault:
        case DISPVIDCTRL_AllianceBrandType_Dacia:
        case DISPVIDCTRL_AllianceBrandType_RSM:
        {
            name = getScreenLayoutsNameForRenault();
            break;
        }
        case DISPVIDCTRL_AllianceBrandType_Infiniti:
        {
            name = "ScreenLayouts_npivi.xml";
            break;
        }
        case DISPVIDCTRL_AllianceBrandType_Nissan:
        case DISPVIDCTRL_AllianceBrandType_Mitsubishi:
        {
            if ((strcmp((const char*)tU8EarlyResolution, "jdi_lam080gx") == 0) || (strcmp((const char*)tU8EarlyResolution, "tianma_tm090adkq01") == 0))
            {
               name = "ScreenLayouts_rnaivi_ST3_HighRes.xml";
               break;
            }
        }
        default:
            break;
    }
    return name;
}

::std::string AIVIConfigurator::getScreenLayoutsNameForRenault() const
{
    ::std::string name = "ScreenLayouts_rivie.xml";
    switch (tU8EarlyDisplayOrientation)
    {
        case DISPVIDCTRL_AllianceDisplayFormatType_Portrait:
        {
            name = "ScreenLayouts_rivieP.xml";
            break;
        }
        default:
            break;
    }
    return name;
}

/* The name in brakes has to be the same as name of a class!!!
 * otherwise plugin will not be accepted*/
DECLARE_LAYERMANAGEMENT_PLUGIN(AIVIConfigurator)
